﻿using System.Collections.Specialized;

using VIRP.EFR.BO;
using VIRP.EFR.DAL.Imaging.Utility;

namespace VIRP.EFR.BLL
{
    internal class AuthService
    {
        private const int CacheTtlMinutes = 15;

        private const string CacheKeyNamespace = "VIRP.EFR.BLL";
        private const string CacheKeyClass = "AuthService";

        private VistaImagingConfigurationService VistaImagingConfigurationService;
        private VistaImagingConfigurationService ConfigurationService
        {
            get
            {
                if (this.VistaImagingConfigurationService == null)
                    this.VistaImagingConfigurationService = new VistaImagingConfigurationService();

                return this.VistaImagingConfigurationService;
            }
        }

        private CertificateService CertServiceRef;
        private CertificateService CertService
        {
            get
            {
                if (this.CertServiceRef == null)
                    this.CertServiceRef = new CertificateService();

                return this.CertServiceRef;
            }
        }

        private ApiTokenValidator TokenValidatorRef;
        private ApiTokenValidator TokenValidator
        {
            get
            {
                if (this.TokenValidatorRef == null)
                    this.TokenValidatorRef = new ApiTokenValidator();

                return this.TokenValidatorRef;
            }
        }

        internal AccessToken GetToken(
            string currentUser, 
            int currentRegistryId, 
            bool? clearCache = null)
        {
            return this.RefreshToken(
                currentUser, 
                currentRegistryId, 
                clearCache);
        }

        private AccessToken RefreshToken(
            string currentUser,
            int currentRegistryId,
            bool? clearCache = null)
        {
            return CacheHelper.GetOrUpdate(GetCacheKey(currentUser),
            (existing) =>
            {
               return clearCache.HasValue && clearCache.Value;
            },
            () =>
            {
                var config = ConfigurationService.GetConfiguration(currentUser, currentRegistryId);

                var tokenResponse = WebRequestor.Get(
                    config.TOKEN_ENDPOINT,
                    CertService.GetCertificate(currentUser, currentRegistryId),
                    currentUser,
                    GetTokenRequestHeaders(currentUser));

                var token = TokenValidator.ValidateAndReturnToken(tokenResponse);

                return new AccessToken(token);
            },
            CacheTtlMinutes);
        }

        internal static NameValueCollection GetAuthRequestHeaders(
            string accessToken,
            string siteName,
            string siteNumber)
        {
            return new NameValueCollection
            {
                { "apiToken", accessToken },
                { "xxx-sitename", siteName },
                { "xxx-sitenumber", siteNumber }
            };
        }

        private static NameValueCollection GetTokenRequestHeaders(string userId)
        {
            string userHeaderValue = null;

            if (!string.IsNullOrEmpty(userId))
            {
                var userIdComponents = userId.Split(new char[] { '\\' });
                if (userIdComponents.Length == 2)
                    userHeaderValue = userIdComponents[1];
                else
                    userHeaderValue = userId;
            }

            return new NameValueCollection
            {
                { "userId", userHeaderValue }
            };
        }

        private static string GetCacheKey(string userId)
        {
            return new CacheKey(CacheKeyNamespace, CacheKeyClass, "GetToken")
                .AddPart(userId)
                .Build();
        }
    }
}
